/**
 * API Route: Single Note
 * PUT /api/notes/[id] - עדכון פתק
 * DELETE /api/notes/[id] - מחיקת פתק
 */

import { NextRequest, NextResponse } from 'next/server';
import { getNotesDatabase } from '@/src/lib/notes-db';

export async function PUT(
    request: NextRequest,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params;
        const body = await request.json();
        const db = await getNotesDatabase();

        const updates: string[] = [];
        const values: any[] = [];

        if (body.listId !== undefined) {
            updates.push('list_id = ?');
            values.push(body.listId);
        }
        if (body.title !== undefined) {
            updates.push('title = ?');
            values.push(body.title);
        }
        if (body.description !== undefined) {
            updates.push('description = ?');
            values.push(body.description);
        }
        if (body.done !== undefined) {
            updates.push('done = ?');
            values.push(body.done ? 1 : 0);
        }
        if (body.important !== undefined) {
            updates.push('important = ?');
            values.push(body.important ? 1 : 0);
        }
        if (body.flagged !== undefined) {
            updates.push('flagged = ?');
            values.push(body.flagged ? 1 : 0);
        }
        if (body.dueDate !== undefined) {
            updates.push('due_date = ?');
            values.push(body.dueDate);
        }
        if (body.reminderDate !== undefined) {
            updates.push('reminder_date = ?');
            values.push(body.reminderDate);
        }
        if (body.reminderTime !== undefined) {
            updates.push('reminder_time = ?');
            values.push(body.reminderTime);
        }
        if (body.reminderRecurrence !== undefined) {
            updates.push('reminder_recurrence = ?');
            values.push(body.reminderRecurrence);
        }
        if (body.subtasks !== undefined) {
            updates.push('subtasks = ?');
            values.push(JSON.stringify(body.subtasks));
        }
        if (body.tags !== undefined) {
            updates.push('tags = ?');
            values.push(JSON.stringify(body.tags));
        }
        if (body.attachments !== undefined) {
            updates.push('attachments = ?');
            values.push(JSON.stringify(body.attachments));
        }

        updates.push('updated_at = ?');
        values.push(new Date().toISOString());
        values.push(id);

        await db.run(
            `UPDATE notes SET ${updates.join(', ')} WHERE id = ?`,
            values
        );

        const row = await db.get('SELECT * FROM notes WHERE id = ?', [id]);

        const note = {
            id: row.id,
            listId: row.list_id,
            title: row.title,
            description: row.description,
            done: row.done === 1,
            important: row.important === 1,
            flagged: row.flagged === 1,
            dueDate: row.due_date,
            reminderDate: row.reminder_date,
            reminderTime: row.reminder_time,
            reminderRecurrence: row.reminder_recurrence || 'none',
            createdAt: row.created_at,
            updatedAt: row.updated_at,
            subtasks: row.subtasks ? JSON.parse(row.subtasks) : [],
            tags: row.tags ? JSON.parse(row.tags) : [],
            attachments: row.attachments ? JSON.parse(row.attachments) : [],
        };

        return NextResponse.json({ note });
    } catch (error: any) {
        console.error('Error updating note:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

export async function DELETE(
    request: NextRequest,
    { params }: { params: Promise<{ id: string }> }
) {
    try {
        const { id } = await params;
        const db = await getNotesDatabase();

        await db.run('DELETE FROM notes WHERE id = ?', [id]);

        return NextResponse.json({ success: true });
    } catch (error: any) {
        console.error('Error deleting note:', error);
        return NextResponse.json(
            { error: error.message },
            { status: 500 }
        );
    }
}

